#region References

using System;
using System.ComponentModel;
using System.Net;
using System.Windows.Forms;

using gov.va.med.vbecs.BOL;
using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL.HL7AL;
using gov.va.med.vbecs.GUI;
using gov.va.med.vbecs.GUI.controls;

using BrRules = gov.va.med.vbecs.Common.VbecsBrokenRules.HL7InterfaceConfig;

#endregion

namespace gov.va.med.vbecs.admin.GUI
{
	/// <summary>
	/// User control encapsulating logic for editing HL7 interface configuration parameters. 
	/// </summary>
	public class EditInterfaceParameters : BaseAdminFormControl
	{	
		#region Inner Classes

		// CR 2940
		private class InterfaceTestResult
		{
			public readonly string Message;
			public readonly int ImageIndex;
			
			private string _errorDetails;
			private bool _isErrorDetailsUpdateable;

			private InterfaceTestResult( StrRes.StringResourceItem message, int imageIndex, bool isErrorDetailsUpdateable, string defaultErrorDetails )
			{
				Message = message.ResString;
				ImageIndex = imageIndex;

				_isErrorDetailsUpdateable = isErrorDetailsUpdateable;
				_errorDetails = defaultErrorDetails;
			}

			public string ErrorDetails
			{
				get
				{
					return _errorDetails;
				}
				set
				{
					if( !_isErrorDetailsUpdateable )
						throw( new InvalidOperationException() );

					_errorDetails = value ?? "No error details is available." ;
				}
			}

			static InterfaceTestResult(){} // getting rid of beforefieldinit

			public static readonly InterfaceTestResult 
				NotTested = new InterfaceTestResult( StrRes.OtherMsg.MUC01.TestResultNotTested(), 0, false, StrRes.Tooltips.MUC04.InterfaceConfigWasNotTested().ResString ),
				Testing = new InterfaceTestResult( StrRes.OtherMsg.MUC01.TestResultTesting(), 1, false, StrRes.Tooltips.MUC04.InterfaceConfigIsBeingTested().ResString ),
				Success = new InterfaceTestResult( StrRes.OtherMsg.MUC01.TestResultSuccessful(), 2, false,  StrRes.Tooltips.MUC04.InterfaceConfigTestSucceeded().ResString ),
				Failure = new InterfaceTestResult( StrRes.OtherMsg.MUC01.TestResultFailed(), 3, true, null );
		}

		#endregion

		#region Variables

		private HL7InterfaceConfig _currentConfig;
		
		private BusinessObjectGuiValidator _hl7GuiValidator;

		private bool _cbInterfaceDisabledCheckedByUser = true;

		private bool _inputDataTypesAreTransferable;
		//CR 2949
		private bool _testConnectionCanBePerformed;
		private bool _interfaceEnabled = false;

		private InterfaceTestResult _connectionTestResult;

        private bool _interfaceChanged = false; // CR 3419
        private bool _serviceIpOrPortChanged = false; // CR 3419

        private bool _upDownChanged = false; // CR 3436

		#endregion

		#region Form Variables

		private IContainer components;

		private ToolTip toolTip1;

// ReSharper disable InconsistentNaming
        private GroupBox grpConfigureInterface;
		private GroupBox grpVBECSApplication;
		private Label lblVBECSIPAddress;
		private TextBox txtVBECSIPAddress;
		private TextBox txtVBECSPort;
		private Label lblVBECSPort;
		private TextBox txtVBECSFacilityID;
		private Label lblVBECSFacilityID;
		private GroupBox grpPurgeCriteria;
		private Label lblErrorDays;
		private Label lblCompletedDays;
		private VbecsNumericUpDown updErrorMessages;
		private Label lblErrorMessages;
		private VbecsNumericUpDown updCompletedMessages;
		private Label lblCompletedMessages;		
		private GroupBox grpRecipient;
		private TextBox txtEmail;
		private Label lblEmail;
		private GroupBox grpInterfacedApplication;
		private TextBox txtInterfacePort;
		private Label lblInterfacePort;
		private TextBox txtInterfaceIPAddress;
		private TextBox txtInterfaceFacilityID;
		private Label lblInterfaceFacilityID;
		private GroupBox grpMessageOptions;
		private VbecsNumericUpDown updReTry;
		private Label lblReTry;
		private Label lblSecs;
		private VbecsNumericUpDown updACKTimeout;
		private Label lblACKTimeout;

		private VbecsButton btnSave;
		private VbecsButton btnClear;
		private GroupBox grpEnableLogging;
		private CheckBox cbLogEvents;
		private TextBox txtInterfaceDomain;
		private RadioButton rdoIpAddress;
		private RadioButton rdoDomain;
		private GroupBox grpConnection;
		private GroupBox grpTestConnection;
		private VbecsButton btnTestConnection;
		private Label lblSuccessIndicator;
		private TextBox txtConnectionTestResults;
		private ImageList imageListInterfaceTestResult;
		private CheckBox cbInterfaceDisabled;
// ReSharper restore InconsistentNaming

		#endregion

		#region Component Designer generated code
		/// <summary> 
		/// Required method for Designer support - do not modify 
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{              
            this.components = new System.ComponentModel.Container();
            System.ComponentModel.ComponentResourceManager resources = new System.ComponentModel.ComponentResourceManager(typeof(EditInterfaceParameters));
            this.grpConfigureInterface = new System.Windows.Forms.GroupBox();
            this.cbInterfaceDisabled = new System.Windows.Forms.CheckBox();
            this.grpEnableLogging = new System.Windows.Forms.GroupBox();
            this.cbLogEvents = new System.Windows.Forms.CheckBox();
            this.grpVBECSApplication = new System.Windows.Forms.GroupBox();
            this.lblVBECSIPAddress = new System.Windows.Forms.Label();
            this.txtVBECSIPAddress = new System.Windows.Forms.TextBox();
            this.txtVBECSPort = new System.Windows.Forms.TextBox();
            this.lblVBECSPort = new System.Windows.Forms.Label();
            this.txtVBECSFacilityID = new System.Windows.Forms.TextBox();
            this.lblVBECSFacilityID = new System.Windows.Forms.Label();
            this.grpPurgeCriteria = new System.Windows.Forms.GroupBox();
            this.lblErrorDays = new System.Windows.Forms.Label();
            this.lblCompletedDays = new System.Windows.Forms.Label();
            this.updErrorMessages = new gov.va.med.vbecs.GUI.controls.VbecsNumericUpDown();
            this.lblErrorMessages = new System.Windows.Forms.Label();
            this.updCompletedMessages = new gov.va.med.vbecs.GUI.controls.VbecsNumericUpDown();
            this.lblCompletedMessages = new System.Windows.Forms.Label();
            this.btnSave = new gov.va.med.vbecs.GUI.controls.VbecsButton();
            this.grpRecipient = new System.Windows.Forms.GroupBox();
            this.txtEmail = new System.Windows.Forms.TextBox();
            this.lblEmail = new System.Windows.Forms.Label();
            this.grpInterfacedApplication = new System.Windows.Forms.GroupBox();
            this.grpTestConnection = new System.Windows.Forms.GroupBox();
            this.btnTestConnection = new gov.va.med.vbecs.GUI.controls.VbecsButton();
            this.lblSuccessIndicator = new System.Windows.Forms.Label();
            this.txtConnectionTestResults = new System.Windows.Forms.TextBox();
            this.grpConnection = new System.Windows.Forms.GroupBox();
            this.rdoIpAddress = new System.Windows.Forms.RadioButton();
            this.txtInterfaceIPAddress = new System.Windows.Forms.TextBox();
            this.txtInterfaceDomain = new System.Windows.Forms.TextBox();
            this.rdoDomain = new System.Windows.Forms.RadioButton();
            this.txtInterfacePort = new System.Windows.Forms.TextBox();
            this.lblInterfacePort = new System.Windows.Forms.Label();
            this.txtInterfaceFacilityID = new System.Windows.Forms.TextBox();
            this.lblInterfaceFacilityID = new System.Windows.Forms.Label();
            this.grpMessageOptions = new System.Windows.Forms.GroupBox();
            this.updReTry = new gov.va.med.vbecs.GUI.controls.VbecsNumericUpDown();
            this.lblReTry = new System.Windows.Forms.Label();
            this.lblSecs = new System.Windows.Forms.Label();
            this.updACKTimeout = new gov.va.med.vbecs.GUI.controls.VbecsNumericUpDown();
            this.lblACKTimeout = new System.Windows.Forms.Label();
            this.btnClear = new gov.va.med.vbecs.GUI.controls.VbecsButton();
            this.imageListInterfaceTestResult = new System.Windows.Forms.ImageList(this.components);
            ((System.ComponentModel.ISupportInitialize)(this.erpMainProvider)).BeginInit();
            this.grpConfigureInterface.SuspendLayout();
            this.grpEnableLogging.SuspendLayout();
            this.grpVBECSApplication.SuspendLayout();
            this.grpPurgeCriteria.SuspendLayout();
            ((System.ComponentModel.ISupportInitialize)(this.updErrorMessages)).BeginInit();
            ((System.ComponentModel.ISupportInitialize)(this.updCompletedMessages)).BeginInit();
            this.grpRecipient.SuspendLayout();
            this.grpInterfacedApplication.SuspendLayout();
            this.grpTestConnection.SuspendLayout();
            this.grpConnection.SuspendLayout();
            this.grpMessageOptions.SuspendLayout();
            ((System.ComponentModel.ISupportInitialize)(this.updReTry)).BeginInit();
            ((System.ComponentModel.ISupportInitialize)(this.updACKTimeout)).BeginInit();
            this.SuspendLayout();
            // 
            // grpConfigureInterface
            // 
            this.grpConfigureInterface.Controls.Add(this.cbInterfaceDisabled);
            this.grpConfigureInterface.Controls.Add(this.grpEnableLogging);
            this.grpConfigureInterface.Controls.Add(this.grpVBECSApplication);
            this.grpConfigureInterface.Controls.Add(this.grpPurgeCriteria);
            this.grpConfigureInterface.Controls.Add(this.btnSave);
            this.grpConfigureInterface.Controls.Add(this.grpRecipient);
            this.grpConfigureInterface.Controls.Add(this.grpInterfacedApplication);
            this.grpConfigureInterface.Controls.Add(this.grpMessageOptions);
            this.grpConfigureInterface.Controls.Add(this.btnClear);
            this.grpConfigureInterface.Dock = System.Windows.Forms.DockStyle.Fill;
            this.grpConfigureInterface.Location = new System.Drawing.Point(0, 0);
            this.grpConfigureInterface.Name = "grpConfigureInterface";
            this.grpConfigureInterface.Size = new System.Drawing.Size(636, 444);
            this.grpConfigureInterface.TabIndex = 2;
            this.grpConfigureInterface.TabStop = false;
            this.grpConfigureInterface.Text = "Configure Interface";
            // 
            // cbInterfaceDisabled
            // 
            this.cbInterfaceDisabled.Location = new System.Drawing.Point(8, 416);
            this.cbInterfaceDisabled.Name = "cbInterfaceDisabled";
            this.cbInterfaceDisabled.Size = new System.Drawing.Size(128, 24);
            this.cbInterfaceDisabled.TabIndex = 23;
            this.cbInterfaceDisabled.Text = "Interface Disabled";
            this.cbInterfaceDisabled.Visible = false;
            this.cbInterfaceDisabled.CheckStateChanged += new System.EventHandler(this.cbInterfaceDisabled_CheckStateChanged);
            // 
            // grpEnableLogging
            // 
            this.grpEnableLogging.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.grpEnableLogging.Controls.Add(this.cbLogEvents);
            this.grpEnableLogging.Location = new System.Drawing.Point(8, 364);
            this.grpEnableLogging.Name = "grpEnableLogging";
            this.grpEnableLogging.Size = new System.Drawing.Size(620, 44);
            this.grpEnableLogging.TabIndex = 22;
            this.grpEnableLogging.TabStop = false;
            this.grpEnableLogging.Text = "Logging Configuration";
            // 
            // cbLogEvents
            // 
            this.cbLogEvents.CheckAlign = System.Drawing.ContentAlignment.MiddleRight;
            this.cbLogEvents.Location = new System.Drawing.Point(8, 16);
            this.cbLogEvents.Name = "cbLogEvents";
            this.cbLogEvents.Size = new System.Drawing.Size(250, 20);
            this.cbLogEvents.TabIndex = 0;
            this.cbLogEvents.Text = "Log Events and HL7 Messages to Event Log";
            this.cbLogEvents.CheckedChanged += new System.EventHandler(this.InputFieldChanged);
            // 
            // grpVBECSApplication
            // 
            this.grpVBECSApplication.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.grpVBECSApplication.Controls.Add(this.lblVBECSIPAddress);
            this.grpVBECSApplication.Controls.Add(this.txtVBECSIPAddress);
            this.grpVBECSApplication.Controls.Add(this.txtVBECSPort);
            this.grpVBECSApplication.Controls.Add(this.lblVBECSPort);
            this.grpVBECSApplication.Controls.Add(this.txtVBECSFacilityID);
            this.grpVBECSApplication.Controls.Add(this.lblVBECSFacilityID);
            this.grpVBECSApplication.Location = new System.Drawing.Point(8, 152);
            this.grpVBECSApplication.Name = "grpVBECSApplication";
            this.grpVBECSApplication.Size = new System.Drawing.Size(620, 64);
            this.grpVBECSApplication.TabIndex = 20;
            this.grpVBECSApplication.TabStop = false;
            this.grpVBECSApplication.Text = "VBECS Application*";
            // 
            // lblVBECSIPAddress
            // 
            this.lblVBECSIPAddress.Location = new System.Drawing.Point(8, 16);
            this.lblVBECSIPAddress.Name = "lblVBECSIPAddress";
            this.lblVBECSIPAddress.Size = new System.Drawing.Size(112, 20);
            this.lblVBECSIPAddress.TabIndex = 10;
            this.lblVBECSIPAddress.Text = "IP Address";
            this.lblVBECSIPAddress.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // txtVBECSIPAddress
            // 
            this.txtVBECSIPAddress.Location = new System.Drawing.Point(120, 16);
            this.txtVBECSIPAddress.MaxLength = 15;
            this.txtVBECSIPAddress.Name = "txtVBECSIPAddress";
            this.txtVBECSIPAddress.Size = new System.Drawing.Size(148, 20);
            this.txtVBECSIPAddress.TabIndex = 4;
            this.txtVBECSIPAddress.TextChanged += new System.EventHandler(this.ServiceConnectivityInputFieldChanged);
            // 
            // txtVBECSPort
            // 
            this.txtVBECSPort.Location = new System.Drawing.Point(356, 16);
            this.txtVBECSPort.MaxLength = 5;
            this.txtVBECSPort.Name = "txtVBECSPort";
            this.txtVBECSPort.Size = new System.Drawing.Size(52, 20);
            this.txtVBECSPort.TabIndex = 5;
            this.txtVBECSPort.TextChanged += new System.EventHandler(this.ServiceConnectivityInputFieldChanged);
            // 
            // lblVBECSPort
            // 
            this.lblVBECSPort.Location = new System.Drawing.Point(288, 16);
            this.lblVBECSPort.Name = "lblVBECSPort";
            this.lblVBECSPort.Size = new System.Drawing.Size(76, 20);
            this.lblVBECSPort.TabIndex = 8;
            this.lblVBECSPort.Text = "Port Number";
            this.lblVBECSPort.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // txtVBECSFacilityID
            // 
            this.txtVBECSFacilityID.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.txtVBECSFacilityID.Location = new System.Drawing.Point(356, 40);
            this.txtVBECSFacilityID.MaxLength = 20;
            this.txtVBECSFacilityID.Name = "txtVBECSFacilityID";
            this.txtVBECSFacilityID.Size = new System.Drawing.Size(252, 20);
            this.txtVBECSFacilityID.TabIndex = 6;
            this.txtVBECSFacilityID.TextChanged += new System.EventHandler(this.InputFieldChanged);
            this.txtVBECSFacilityID.MouseHover += new System.EventHandler(this.txtVBECSFacilityID_MouseHover);
            // 
            // lblVBECSFacilityID
            // 
            this.lblVBECSFacilityID.Location = new System.Drawing.Point(288, 40);
            this.lblVBECSFacilityID.Name = "lblVBECSFacilityID";
            this.lblVBECSFacilityID.Size = new System.Drawing.Size(64, 20);
            this.lblVBECSFacilityID.TabIndex = 5;
            this.lblVBECSFacilityID.Text = "Facility ID";
            this.lblVBECSFacilityID.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // grpPurgeCriteria
            // 
            this.grpPurgeCriteria.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.grpPurgeCriteria.Controls.Add(this.lblErrorDays);
            this.grpPurgeCriteria.Controls.Add(this.lblCompletedDays);
            this.grpPurgeCriteria.Controls.Add(this.updErrorMessages);
            this.grpPurgeCriteria.Controls.Add(this.lblErrorMessages);
            this.grpPurgeCriteria.Controls.Add(this.updCompletedMessages);
            this.grpPurgeCriteria.Controls.Add(this.lblCompletedMessages);
            this.grpPurgeCriteria.Location = new System.Drawing.Point(8, 268);
            this.grpPurgeCriteria.Name = "grpPurgeCriteria";
            this.grpPurgeCriteria.Size = new System.Drawing.Size(620, 44);
            this.grpPurgeCriteria.TabIndex = 16;
            this.grpPurgeCriteria.TabStop = false;
            this.grpPurgeCriteria.Text = "Purge Criteria";
            // 
            // lblErrorDays
            // 
            this.lblErrorDays.Location = new System.Drawing.Point(384, 16);
            this.lblErrorDays.Name = "lblErrorDays";
            this.lblErrorDays.Size = new System.Drawing.Size(28, 20);
            this.lblErrorDays.TabIndex = 5;
            this.lblErrorDays.Text = "days";
            this.lblErrorDays.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // lblCompletedDays
            // 
            this.lblCompletedDays.Location = new System.Drawing.Point(164, 16);
            this.lblCompletedDays.Name = "lblCompletedDays";
            this.lblCompletedDays.Size = new System.Drawing.Size(28, 20);
            this.lblCompletedDays.TabIndex = 4;
            this.lblCompletedDays.Text = "days";
            this.lblCompletedDays.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // updErrorMessages
            // 
            this.updErrorMessages.Location = new System.Drawing.Point(336, 16);
            this.updErrorMessages.Maximum = new decimal(new int[] {
            30,
            0,
            0,
            0});
            this.updErrorMessages.Minimum = new decimal(new int[] {
            1,
            0,
            0,
            0});
            this.updErrorMessages.Name = "updErrorMessages";
            this.updErrorMessages.Size = new System.Drawing.Size(44, 20);
            this.updErrorMessages.TabIndex = 10;
            this.updErrorMessages.Value = new decimal(new int[] {
            7,
            0,
            0,
            0});
            this.updErrorMessages.ValueChanged += new System.EventHandler(this.NumericUpDownValueChanged);
            this.updErrorMessages.KeyUp += new System.Windows.Forms.KeyEventHandler(this.NumericUpDown_KeyUp);
            this.updErrorMessages.Leave += new System.EventHandler(this.InputFieldChanged);
            // 
            // lblErrorMessages
            // 
            this.lblErrorMessages.Location = new System.Drawing.Point(224, 16);
            this.lblErrorMessages.Name = "lblErrorMessages";
            this.lblErrorMessages.Size = new System.Drawing.Size(104, 20);
            this.lblErrorMessages.TabIndex = 3;
            this.lblErrorMessages.Text = "Messages in Error*";
            this.lblErrorMessages.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // updCompletedMessages
            // 
            this.updCompletedMessages.Location = new System.Drawing.Point(120, 16);
            this.updCompletedMessages.Maximum = new decimal(new int[] {
            30,
            0,
            0,
            0});
            this.updCompletedMessages.Minimum = new decimal(new int[] {
            1,
            0,
            0,
            0});
            this.updCompletedMessages.Name = "updCompletedMessages";
            this.updCompletedMessages.Size = new System.Drawing.Size(44, 20);
            this.updCompletedMessages.TabIndex = 9;
            this.updCompletedMessages.Value = new decimal(new int[] {
            7,
            0,
            0,
            0});
            this.updCompletedMessages.ValueChanged += new System.EventHandler(this.NumericUpDownValueChanged);
            this.updCompletedMessages.KeyUp += new System.Windows.Forms.KeyEventHandler(this.NumericUpDown_KeyUp);
            this.updCompletedMessages.Leave += new System.EventHandler(this.InputFieldChanged);
            // 
            // lblCompletedMessages
            // 
            this.lblCompletedMessages.Location = new System.Drawing.Point(8, 16);
            this.lblCompletedMessages.Name = "lblCompletedMessages";
            this.lblCompletedMessages.Size = new System.Drawing.Size(120, 20);
            this.lblCompletedMessages.TabIndex = 2;
            this.lblCompletedMessages.Text = "Completed Messages*";
            this.lblCompletedMessages.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // btnSave
            // 
            this.btnSave.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Right)));
            this.btnSave.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((byte)(0)));
            this.btnSave.EnabledButton = false;
            this.btnSave.Image = null;
            this.btnSave.Location = new System.Drawing.Point(552, 416);
            this.btnSave.Name = "btnSave";
            this.btnSave.Size = new System.Drawing.Size(75, 23);
            this.btnSave.TabIndex = 15;
            this.btnSave.Click += new System.EventHandler(this.btnSave_Click);
            // 
            // grpRecipient
            // 
            this.grpRecipient.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.grpRecipient.Controls.Add(this.txtEmail);
            this.grpRecipient.Controls.Add(this.lblEmail);
            this.grpRecipient.Location = new System.Drawing.Point(8, 316);
            this.grpRecipient.Name = "grpRecipient";
            this.grpRecipient.Size = new System.Drawing.Size(620, 44);
            this.grpRecipient.TabIndex = 13;
            this.grpRecipient.TabStop = false;
            this.grpRecipient.Text = "Interface Failure Alert Recipient";
            // 
            // txtEmail
            // 
            this.txtEmail.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.txtEmail.Location = new System.Drawing.Point(120, 16);
            this.txtEmail.MaxLength = 255;
            this.txtEmail.Name = "txtEmail";
            this.txtEmail.Size = new System.Drawing.Size(488, 20);
            this.txtEmail.TabIndex = 11;
            this.txtEmail.TextChanged += new System.EventHandler(this.InputFieldChanged);
            // 
            // lblEmail
            // 
            this.lblEmail.Location = new System.Drawing.Point(8, 16);
            this.lblEmail.Name = "lblEmail";
            this.lblEmail.Size = new System.Drawing.Size(108, 20);
            this.lblEmail.TabIndex = 12;
            this.lblEmail.Text = "E-mail Address*";
            this.lblEmail.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // grpInterfacedApplication
            // 
            this.grpInterfacedApplication.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.grpInterfacedApplication.Controls.Add(this.grpTestConnection);
            this.grpInterfacedApplication.Controls.Add(this.grpConnection);
            this.grpInterfacedApplication.Controls.Add(this.txtInterfacePort);
            this.grpInterfacedApplication.Controls.Add(this.lblInterfacePort);
            this.grpInterfacedApplication.Controls.Add(this.txtInterfaceFacilityID);
            this.grpInterfacedApplication.Controls.Add(this.lblInterfaceFacilityID);
            this.grpInterfacedApplication.Location = new System.Drawing.Point(8, 16);
            this.grpInterfacedApplication.Name = "grpInterfacedApplication";
            this.grpInterfacedApplication.Size = new System.Drawing.Size(620, 132);
            this.grpInterfacedApplication.TabIndex = 19;
            this.grpInterfacedApplication.TabStop = false;
            this.grpInterfacedApplication.Text = "Interfaced Application*";
            // 
            // grpTestConnection
            // 
            this.grpTestConnection.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.grpTestConnection.Controls.Add(this.btnTestConnection);
            this.grpTestConnection.Controls.Add(this.lblSuccessIndicator);
            this.grpTestConnection.Controls.Add(this.txtConnectionTestResults);
            this.grpTestConnection.Location = new System.Drawing.Point(8, 88);
            this.grpTestConnection.Name = "grpTestConnection";
            this.grpTestConnection.Size = new System.Drawing.Size(600, 40);
            this.grpTestConnection.TabIndex = 18;
            this.grpTestConnection.TabStop = false;
            // 
            // btnTestConnection
            // 
            this.btnTestConnection.EnabledButton = false;
            this.btnTestConnection.Image = ((System.Drawing.Image)(resources.GetObject("btnTestConnection.Image")));
            this.btnTestConnection.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((byte)(0)));
            this.btnTestConnection.Location = new System.Drawing.Point(8, 12);
            this.btnTestConnection.Name = "btnTestConnection";
            this.btnTestConnection.Size = new System.Drawing.Size(116, 23);
            this.btnTestConnection.TabIndex = 7;
            this.btnTestConnection.Click += new System.EventHandler(this.btnTestConnection_Click);
            // 
            // lblSuccessIndicator
            // 
            this.lblSuccessIndicator.BorderStyle = System.Windows.Forms.BorderStyle.FixedSingle;
            this.lblSuccessIndicator.Location = new System.Drawing.Point(130, 14);
            this.lblSuccessIndicator.Name = "lblSuccessIndicator";
            this.lblSuccessIndicator.Size = new System.Drawing.Size(20, 20);
            this.lblSuccessIndicator.TabIndex = 5;
            this.lblSuccessIndicator.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // txtConnectionTestResults
            // 
            this.txtConnectionTestResults.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.txtConnectionTestResults.Location = new System.Drawing.Point(152, 14);
            this.txtConnectionTestResults.Name = "txtConnectionTestResults";
            this.txtConnectionTestResults.ReadOnly = true;
            this.txtConnectionTestResults.Size = new System.Drawing.Size(440, 20);
            this.txtConnectionTestResults.TabIndex = 5;
            this.txtConnectionTestResults.TabStop = false;
            this.txtConnectionTestResults.Text = "Not tested";
            // 
            // grpConnection
            // 
            this.grpConnection.Controls.Add(this.rdoIpAddress);
            this.grpConnection.Controls.Add(this.txtInterfaceIPAddress);
            this.grpConnection.Controls.Add(this.txtInterfaceDomain);
            this.grpConnection.Controls.Add(this.rdoDomain);
            this.grpConnection.Location = new System.Drawing.Point(8, 16);
            this.grpConnection.Name = "grpConnection";
            this.grpConnection.Size = new System.Drawing.Size(272, 72);
            this.grpConnection.TabIndex = 14;
            this.grpConnection.TabStop = false;
            this.grpConnection.Text = "Connection Method";
            // 
            // rdoIpAddress
            // 
            this.rdoIpAddress.Location = new System.Drawing.Point(16, 16);
            this.rdoIpAddress.Name = "rdoIpAddress";
            this.rdoIpAddress.Size = new System.Drawing.Size(80, 24);
            this.rdoIpAddress.TabIndex = 12;
            this.rdoIpAddress.Text = "IP Address";
            this.rdoIpAddress.CheckedChanged += new System.EventHandler(this.rdo_CheckedChanged);
            // 
            // txtInterfaceIPAddress
            // 
            this.txtInterfaceIPAddress.Location = new System.Drawing.Point(112, 16);
            this.txtInterfaceIPAddress.MaxLength = 15;
            this.txtInterfaceIPAddress.Name = "txtInterfaceIPAddress";
            this.txtInterfaceIPAddress.Size = new System.Drawing.Size(148, 20);
            this.txtInterfaceIPAddress.TabIndex = 1;
            this.txtInterfaceIPAddress.TextChanged += new System.EventHandler(this.InterfaceConnectivityInputFieldChanged);
            // 
            // txtInterfaceDomain
            // 
            this.txtInterfaceDomain.Location = new System.Drawing.Point(112, 40);
            this.txtInterfaceDomain.MaxLength = 100;
            this.txtInterfaceDomain.Name = "txtInterfaceDomain";
            this.txtInterfaceDomain.Size = new System.Drawing.Size(148, 20);
            this.txtInterfaceDomain.TabIndex = 10;
            this.txtInterfaceDomain.TextChanged += new System.EventHandler(this.InterfaceConnectivityInputFieldChanged);
            this.txtInterfaceDomain.MouseHover += new System.EventHandler(this.txtInterfaceDomain_MouseHover);
            // 
            // rdoDomain
            // 
            this.rdoDomain.Location = new System.Drawing.Point(16, 40);
            this.rdoDomain.Name = "rdoDomain";
            this.rdoDomain.Size = new System.Drawing.Size(104, 24);
            this.rdoDomain.TabIndex = 13;
            this.rdoDomain.Text = "Domain";
            this.rdoDomain.CheckedChanged += new System.EventHandler(this.rdo_CheckedChanged);
            // 
            // txtInterfacePort
            // 
            this.txtInterfacePort.Location = new System.Drawing.Point(356, 32);
            this.txtInterfacePort.MaxLength = 5;
            this.txtInterfacePort.Name = "txtInterfacePort";
            this.txtInterfacePort.Size = new System.Drawing.Size(52, 20);
            this.txtInterfacePort.TabIndex = 2;
            this.txtInterfacePort.TextChanged += new System.EventHandler(this.InterfaceConnectivityInputFieldChanged);
            // 
            // lblInterfacePort
            // 
            this.lblInterfacePort.Location = new System.Drawing.Point(288, 32);
            this.lblInterfacePort.Name = "lblInterfacePort";
            this.lblInterfacePort.Size = new System.Drawing.Size(76, 20);
            this.lblInterfacePort.TabIndex = 6;
            this.lblInterfacePort.Text = "Port Number";
            this.lblInterfacePort.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // txtInterfaceFacilityID
            // 
            this.txtInterfaceFacilityID.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.txtInterfaceFacilityID.Location = new System.Drawing.Point(356, 56);
            this.txtInterfaceFacilityID.MaxLength = 20;
            this.txtInterfaceFacilityID.Name = "txtInterfaceFacilityID";
            this.txtInterfaceFacilityID.Size = new System.Drawing.Size(252, 20);
            this.txtInterfaceFacilityID.TabIndex = 3;
            this.txtInterfaceFacilityID.TextChanged += new System.EventHandler(this.InputFieldChanged);
            this.txtInterfaceFacilityID.MouseHover += new System.EventHandler(this.txtInterfaceFacilityID_MouseHover);
            // 
            // lblInterfaceFacilityID
            // 
            this.lblInterfaceFacilityID.Location = new System.Drawing.Point(288, 56);
            this.lblInterfaceFacilityID.Name = "lblInterfaceFacilityID";
            this.lblInterfaceFacilityID.Size = new System.Drawing.Size(64, 20);
            this.lblInterfaceFacilityID.TabIndex = 9;
            this.lblInterfaceFacilityID.Text = "Facility ID";
            this.lblInterfaceFacilityID.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // grpMessageOptions
            // 
            this.grpMessageOptions.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
            | System.Windows.Forms.AnchorStyles.Right)));
            this.grpMessageOptions.Controls.Add(this.updReTry);
            this.grpMessageOptions.Controls.Add(this.lblReTry);
            this.grpMessageOptions.Controls.Add(this.lblSecs);
            this.grpMessageOptions.Controls.Add(this.updACKTimeout);
            this.grpMessageOptions.Controls.Add(this.lblACKTimeout);
            this.grpMessageOptions.Location = new System.Drawing.Point(8, 220);
            this.grpMessageOptions.Name = "grpMessageOptions";
            this.grpMessageOptions.Size = new System.Drawing.Size(620, 44);
            this.grpMessageOptions.TabIndex = 21;
            this.grpMessageOptions.TabStop = false;
            this.grpMessageOptions.Text = "Message Options";
            // 
            // updReTry
            // 
            this.updReTry.Location = new System.Drawing.Point(336, 16);
            this.updReTry.Maximum = new decimal(new int[] {
            99,
            0,
            0,
            0});
            this.updReTry.Minimum = new decimal(new int[] {
            1,
            0,
            0,
            0});
            this.updReTry.Name = "updReTry";
            this.updReTry.Size = new System.Drawing.Size(44, 20);
            this.updReTry.TabIndex = 8;
            this.updReTry.Value = new decimal(new int[] {
            8,
            0,
            0,
            0});
            this.updReTry.ValueChanged += new System.EventHandler(this.NumericUpDownValueChanged);
            this.updReTry.KeyUp += new System.Windows.Forms.KeyEventHandler(this.NumericUpDown_KeyUp);
            this.updReTry.Leave += new System.EventHandler(this.InputFieldChanged);
            // 
            // lblReTry
            // 
            this.lblReTry.Location = new System.Drawing.Point(224, 16);
            this.lblReTry.Name = "lblReTry";
            this.lblReTry.Size = new System.Drawing.Size(120, 20);
            this.lblReTry.TabIndex = 9;
            this.lblReTry.Text = "Re-Transmit Attempts*";
            this.lblReTry.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // lblSecs
            // 
            this.lblSecs.Location = new System.Drawing.Point(164, 16);
            this.lblSecs.Name = "lblSecs";
            this.lblSecs.Size = new System.Drawing.Size(28, 20);
            this.lblSecs.TabIndex = 8;
            this.lblSecs.Text = "secs";
            this.lblSecs.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // updACKTimeout
            // 
            this.updACKTimeout.Location = new System.Drawing.Point(120, 16);
            this.updACKTimeout.Maximum = new decimal(new int[] {
            999,
            0,
            0,
            0});
            this.updACKTimeout.Minimum = new decimal(new int[] {
            1,
            0,
            0,
            0});
            this.updACKTimeout.Name = "updACKTimeout";
            this.updACKTimeout.Size = new System.Drawing.Size(44, 20);
            this.updACKTimeout.TabIndex = 7;
            this.updACKTimeout.Value = new decimal(new int[] {
            1,
            0,
            0,
            0});
            this.updACKTimeout.ValueChanged += new System.EventHandler(this.NumericUpDownValueChanged);
            this.updACKTimeout.KeyUp += new System.Windows.Forms.KeyEventHandler(this.NumericUpDown_KeyUp);
            this.updACKTimeout.Leave += new System.EventHandler(this.InputFieldChanged);
            // 
            // lblACKTimeout
            // 
            this.lblACKTimeout.Location = new System.Drawing.Point(8, 16);
            this.lblACKTimeout.Name = "lblACKTimeout";
            this.lblACKTimeout.Size = new System.Drawing.Size(100, 20);
            this.lblACKTimeout.TabIndex = 6;
            this.lblACKTimeout.Text = "ACK Timeout*";
            this.lblACKTimeout.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
            // 
            // btnClear
            // 
            this.btnClear.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Bottom | System.Windows.Forms.AnchorStyles.Right)));
            this.btnClear.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((byte)(0))); 
            this.btnClear.Image = null;
            this.btnClear.Location = new System.Drawing.Point(468, 416);
            this.btnClear.Name = "btnClear";
            this.btnClear.Size = new System.Drawing.Size(75, 23);
            this.btnClear.TabIndex = 19;
            this.btnClear.Click += new System.EventHandler(this.btnClear_Click);
            // 
            // imageListInterfaceTestResult
            // 
            this.imageListInterfaceTestResult.ImageStream = ((System.Windows.Forms.ImageListStreamer)(resources.GetObject("imageListInterfaceTestResult.ImageStream")));
            this.imageListInterfaceTestResult.TransparentColor = System.Drawing.Color.Transparent;
            this.imageListInterfaceTestResult.Images.SetKeyName(0, "");
            this.imageListInterfaceTestResult.Images.SetKeyName(1, "");
            this.imageListInterfaceTestResult.Images.SetKeyName(2, "");
            this.imageListInterfaceTestResult.Images.SetKeyName(3, "");
            // 
            // EditInterfaceParameters
            // 
            this.Controls.Add(this.grpConfigureInterface);
        
            
            
              
        
           
            this.Name = "EditInterfaceParameters";
            this.Size = new System.Drawing.Size(636, 444);
            ((System.ComponentModel.ISupportInitialize)(this.erpMainProvider)).EndInit();
            this.grpConfigureInterface.ResumeLayout(false);
            this.grpEnableLogging.ResumeLayout(false);
            this.grpVBECSApplication.ResumeLayout(false);
            this.grpVBECSApplication.PerformLayout();
            this.grpPurgeCriteria.ResumeLayout(false);
            ((System.ComponentModel.ISupportInitialize)(this.updErrorMessages)).EndInit();
            ((System.ComponentModel.ISupportInitialize)(this.updCompletedMessages)).EndInit();
            this.grpRecipient.ResumeLayout(false);
            this.grpRecipient.PerformLayout();
            this.grpInterfacedApplication.ResumeLayout(false);
            this.grpInterfacedApplication.PerformLayout();
            this.grpTestConnection.ResumeLayout(false);
            this.grpTestConnection.PerformLayout();
            this.grpConnection.ResumeLayout(false);
            this.grpConnection.PerformLayout();
            this.grpMessageOptions.ResumeLayout(false);
            ((System.ComponentModel.ISupportInitialize)(this.updReTry)).EndInit();
            ((System.ComponentModel.ISupportInitialize)(this.updACKTimeout)).EndInit();
            this.ResumeLayout(false);

		}
		#endregion

		#region Constructor / Destructor

		/// <summary>
		/// 
		/// </summary>
		public EditInterfaceParameters() : base()
		{
			components = new Container();

			// This call is required by the Windows.Forms Form Designer.
			InitializeComponent();

			toolTip1 = new ToolTip(components);

			btnTestConnection.EnabledButton = false;
			lblSuccessIndicator.Enabled = false;
			txtConnectionTestResults.Enabled = false;

			lblSuccessIndicator.Image = imageListInterfaceTestResult.Images[0];

			_inputDataTypesAreTransferable = false;
			//CR 2949
			_testConnectionCanBePerformed = false;
			btnSave.Text = "&Save";
			btnClear.Text = "&Clear";
			btnTestConnection.Text = " &Test Connection";

            _interfaceChanged = false; // CR 3419
            _serviceIpOrPortChanged = false; // CR 3419
            _upDownChanged = false; // CR 3436
            

			BindHl7InterfaceValidator();
		}

		/// <summary> 
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#endregion

		/// <summary>
		/// 
		/// </summary>
		private void ResetStateVars()
		{
			DataLoadMode.Release();
			//
			_connectionTestResult = InterfaceTestResult.NotTested;
		}

		/// <summary>
		/// CR 2940
		/// </summary>
		private void btnTestConnection_Click(object sender, EventArgs e)
		{
			_connectionTestResult = InterfaceTestResult.Testing;			
		
			using (new WaitCursor())
			{
				_connectionTestResult = TestConnection();
			}
			
			UpdateSaveButtonToolTip();
			UpdateSaveButtonState();

			//
			UpdateTestConnectionButton();
			//
			UpdateTestResults();
			//
			grpTestConnection.Enabled = btnTestConnection.EnabledButton;
			//
			Refresh();
		}

		/// <summary>
		/// CR 2940
		/// </summary>
		private InterfaceTestResult TestConnection()
		{
			HL7TestConnection testConnection = null;
			//
			InterfaceTestResult result = InterfaceTestResult.Failure;
			//
			try
			{
				if ( txtInterfaceIPAddress.Text != null && txtInterfaceIPAddress.Text != string.Empty 
					&& txtInterfacePort.Text != null && txtInterfacePort.Text != string.Empty )
				{
					testConnection = new HL7TestConnection( IPAddress.Parse(txtInterfaceIPAddress.Text.Trim()), Convert.ToInt32(txtInterfacePort.Text.Trim()) );
				}
				else if ( txtInterfaceDomain.Text != null && txtInterfaceDomain.Text != string.Empty 
					&& txtInterfacePort.Text != null && txtInterfacePort.Text != string.Empty )
				{
					testConnection = new HL7TestConnection( txtInterfaceDomain.Text, Convert.ToInt32(txtInterfacePort.Text) );
				}
				//
				if ( testConnection != null )
				{
					testConnection.Connect();
					//
					if ( testConnection.IsConnected )
					{
						result = InterfaceTestResult.Success;
						//
						_currentConfig.Interface.ConnectedTested = true;
					}
				}
			}
			catch(Exception ex)
			{
				result.ErrorDetails = ex.ToString();
			}
			finally
			{
				if ( testConnection != null )
				{
					testConnection.Close();
				}
			}
			//
			return result;
		}

		/// <summary>
		/// CR 2940
		/// </summary>
		private bool AllowConnectionTest()
		{
			return (_currentConfig != null && (_currentConfig.Interface.InterfaceIPAddress != null 
				|| _currentConfig.Interface.InterfaceDomainName != null)) &&
				(( txtInterfaceIPAddress.Text != null && txtInterfaceIPAddress.Text != string.Empty 
				&& txtInterfacePort.Text != null && txtInterfacePort.Text != string.Empty )
				|| ( txtInterfaceDomain.Text != null && txtInterfaceDomain.Text != string.Empty 
				&& txtInterfacePort.Text != null && txtInterfacePort.Text != string.Empty ));
		}

		private void UpdateButtons()
		{
			UpdateSaveButtonState();
			//
			UpdateSaveButtonToolTip();
			//
			UpdateClearButton();
			//
			UpdateTestConnectionButton();
		}


		/// <summary>
		/// 
		/// </summary>
		private void UpdateClearButton()
		{
			UpdateOnFormClearButton( btnClear, _currentConfig.IsDirty );
		}

		/// <summary>
		/// CR 2940
		/// </summary>
		private void UpdateTestConnectionButton()
		{
			if (_currentConfig.Interface.InterfaceName == InterfaceName.Patient_Merge.ToString() || 
				_currentConfig.Interface.InterfaceName == InterfaceName.Patient_Update.ToString() ||
                _currentConfig.Interface.InterfaceName == InterfaceName.Auto_Instrument.ToString())
			{
				btnTestConnection.EnabledButton = false;
				lblSuccessIndicator.Enabled = false;
				btnTestConnection.SetToolTip("");
			}
			else
			{
				//CR 2952, CR 2954 fixed tooltip for TestConnection button
				if( _connectionTestResult == InterfaceTestResult.NotTested ||
					_connectionTestResult == InterfaceTestResult.Success ||
					_connectionTestResult == InterfaceTestResult.Failure)
				{
					//CR 2949
					btnTestConnection.EnabledButton = _testConnectionCanBePerformed;					
					btnTestConnection.SetToolTip( ( _testConnectionCanBePerformed ? StrRes.Tooltips.MUC04.ClickToTestInterfaceConfig() : StrRes.Tooltips.MUC04.UnableToTestSinceInterfaceConfigIsInvalid() ).ResString );
					lblSuccessIndicator.Enabled = _testConnectionCanBePerformed;
					if (_testConnectionCanBePerformed)
						SetToolTip( lblSuccessIndicator, StrRes.Tooltips.MUC04.InterfaceConfigWasNotTested() );
				} 
				else if( _connectionTestResult == InterfaceTestResult.Testing )
				{
					btnTestConnection.EnabledButton = false;					
					btnTestConnection.SetToolTip( StrRes.Tooltips.MUC04.InterfaceConfigIsBeingTested().ResString );
					lblSuccessIndicator.Enabled = false;
				}
			}
		}

		/// <summary>
		/// CR 2940
		/// </summary>
		private void UpdateTestResults()
		{
			txtConnectionTestResults.Text = _connectionTestResult.Message;
			lblSuccessIndicator.Image = imageListInterfaceTestResult.Images[ _connectionTestResult.ImageIndex ];			

			SetToolTip( txtConnectionTestResults, _connectionTestResult.ErrorDetails );
			SetToolTip( lblSuccessIndicator, _connectionTestResult.ErrorDetails );
		}

		/// <summary>
		/// 
		/// </summary>
		private void BindHl7InterfaceValidator()
		{
			_hl7GuiValidator = new BusinessObjectGuiValidator( erpMainProvider, BrRules.FormClassName );
			//
			_hl7GuiValidator.SetControlBindings( txtVBECSIPAddress, BrRules.VbecsIPAddressNotSet );
			_hl7GuiValidator.SetControlBindings( txtVBECSPort, BrRules.VbecsPortNumberNotSet, BrRules.VbecsPortNumberOutOfRange );
			_hl7GuiValidator.SetControlBindings( txtInterfaceFacilityID, BrRules.InterfaceFacilityIdsMatch );
			_hl7GuiValidator.SetControlBindings( txtVBECSFacilityID, BrRules.InterfaceFacilityIdsMatch );
			_hl7GuiValidator.SetControlBindings( updACKTimeout, BrRules.AckTimeoutNotSet, BrRules.AckTimeoutOutOfRange );
			_hl7GuiValidator.SetControlBindings( updReTry, BrRules.ReTransmitAttemptsNotSet, BrRules.ReTransmitAttemptsOutOfRange );				
			_hl7GuiValidator.SetControlBindings( updCompletedMessages, BrRules.PurgeSuccessfulMessagesNotSet, BrRules.PurgeSuccessfulMessageOutOfRange );
			_hl7GuiValidator.SetControlBindings( updErrorMessages, BrRules.PurgeUnSuccessfulMessageNotSet, BrRules.PurgeUnSuccessfulMessageOutOfRange );
			_hl7GuiValidator.SetControlBindings( txtEmail, BrRules.InterfaceAdministratorEmailNotSet, BrRules.InterfaceAdministratorEmailInvalid );
		}

		/// <summary>
		/// CR 2940
		/// </summary>
		public void SwitchToInterfaceConfig( HL7InterfaceConfig hl7InterfaceConfig )
		{
			if( hl7InterfaceConfig == null )
			{
				throw( new ArgumentNullException( "hl7InterfaceConfig" ) );
			}
			//
			_currentConfig = hl7InterfaceConfig;
			//
			//CR 2952, CR 2954
			_validator.ClearValidationError(txtInterfaceFacilityID);
			_validator.ClearValidationError(txtVBECSFacilityID);
			SetToolTip( txtConnectionTestResults, "" );
			SetToolTip( lblSuccessIndicator, "" );
			//
			DisplayInterfaceData();
			//
			if( _currentConfig != null && (_currentConfig.Interface.InterfaceIPAddress == null 
				&& _currentConfig.Interface.InterfaceDomainName == null))
			{
				grpInterfacedApplication.Text = "Interfaced Application";
				lblInterfaceFacilityID.Text = "Facility ID*";
			}
			else if( _currentConfig != null && (_currentConfig.Interface.InterfaceIPAddress != null 
				|| _currentConfig.Interface.InterfaceDomainName != null))
			{
				grpInterfacedApplication.Text = "Interfaced Application*";
				lblInterfaceFacilityID.Text = "Facility ID";
			}
			//
			if ( !_currentConfig.Interface.ConnectedTested )
			{
				lblSuccessIndicator.Image = imageListInterfaceTestResult.Images[0];
				txtConnectionTestResults.Text = string.Empty;
			}
			//
			DoInputDataTypeCheck();
			UpdateButtons();
		}

		/// <summary>
		/// 
		/// </summary>
		private void DisplayInterfaceData()
		{
			DataLoadMode.Lock();

			HL7Interface _interface = _currentConfig.Interface;

			//CR 2940
			if ( _interface.InterfaceName == InterfaceName.BCE_COTS.ToString() ||
                 _interface.InterfaceName == InterfaceName.Auto_Instrument.ToString())
			{

				if (_interface.InterfaceActiveIndicator == true)
				{
					grpInterfacedApplication.Enabled = true;
					grpVBECSApplication.Enabled = true;
					grpMessageOptions.Enabled = true;
					grpPurgeCriteria.Enabled = true;
					grpRecipient.Enabled = true;
					grpEnableLogging.Enabled = true;
					btnClear.EnabledButton = true;
					btnSave.EnabledButton = false;

					cbInterfaceDisabled.Visible = true;
					cbInterfaceDisabled.Enabled = true;
					if (cbInterfaceDisabled.CheckState != CheckState.Unchecked)
					{
						_cbInterfaceDisabledCheckedByUser = false;
						cbInterfaceDisabled.CheckState = CheckState.Unchecked;
					}
				}
				else
				{
					grpInterfacedApplication.Enabled = false;
					grpVBECSApplication.Enabled = false;
					grpMessageOptions.Enabled = false;
					grpPurgeCriteria.Enabled = false;
					grpRecipient.Enabled = false;
					grpEnableLogging.Enabled = false;
					btnClear.EnabledButton = false;
					btnSave.EnabledButton = false;

					cbInterfaceDisabled.Visible = true;
					cbInterfaceDisabled.Enabled = true;
					if (cbInterfaceDisabled.CheckState != CheckState.Checked)
					{
						_cbInterfaceDisabledCheckedByUser = false;
						cbInterfaceDisabled.CheckState = CheckState.Checked;
					}
				}
			}
			else
			{
				grpInterfacedApplication.Enabled = true;
				grpVBECSApplication.Enabled = true;
				grpMessageOptions.Enabled = true;
				grpPurgeCriteria.Enabled = true;
				grpRecipient.Enabled = true;
				grpEnableLogging.Enabled = true;
				btnClear.EnabledButton = true;
				btnSave.EnabledButton = false;
			}

			txtInterfaceIPAddress.Text = StUiNullConvert.From( _interface.InterfaceIPAddress );
			if (txtInterfaceIPAddress.Text.Trim().Length > 0) rdoIpAddress.Checked = true;
			
			txtInterfaceDomain.Text = StUiNullConvert.From( _interface.InterfaceDomainName );
			if (txtInterfaceDomain.Text.Trim().Length > 0) rdoDomain.Checked = true;

			if (!rdoIpAddress.Checked && !rdoDomain.Checked)
			{
				rdoIpAddress.Checked = true;
			}

			txtInterfacePort.Text = StUiNullConvert.From( _interface.InterfacePortNumber );
			txtInterfaceFacilityID.Text = StUiNullConvert.From( _interface.InterfaceFacilityId );

			txtVBECSIPAddress.Text = StUiNullConvert.From( _interface.VbecsIPAddress );
			txtVBECSPort.Text = StUiNullConvert.From( _interface.VbecsPortNumber );
			txtVBECSFacilityID.Text = StUiNullConvert.From( _interface.VbecsFacilityId );			

			updACKTimeout.Value = StUiNullConvert.DecimalFrom( _interface.AckTimeout, 1 );
			updReTry.Value = StUiNullConvert.DecimalFrom( _interface.ReTransmitAttempts, 1 );

			updCompletedMessages.Value = StUiNullConvert.DecimalFrom( _interface.PurgeSuccessfulMessage, 1 );
			updErrorMessages.Value = StUiNullConvert.DecimalFrom( _interface.PurgeUnSuccessfulMessage, 1  );


           

			txtEmail.Text = StUiNullConvert.From( _interface.InterfaceAdministratorEmail );

			cbLogEvents.Checked = _interface.LogEvents;

			AdjustFieldsState();

            _interfaceChanged = false; // CR 3419
            _serviceIpOrPortChanged = false; // CR 3419
            _upDownChanged = false; // CR 3436

			DataLoadMode.Release();
		}

		/// <summary>
		/// CR 2940
		/// </summary>
		private void AdjustFieldsState()
		{
			//CR 2946 Added check for interface name
			bool disableIpAddressAndDomainEditing = ( _currentConfig.Interface.InterfaceIPAddress == null && _currentConfig.Interface.InterfaceDomainName == null &&
                _currentConfig.Interface.InterfaceName == InterfaceName.Patient_Merge.ToString() || _currentConfig.Interface.InterfaceName == InterfaceName.Patient_Update.ToString() || _currentConfig.Interface.InterfaceName == InterfaceName.Auto_Instrument.ToString());
			//
			if (disableIpAddressAndDomainEditing)
			{
				rdoIpAddress.Enabled = false;
				rdoDomain.Enabled = false;
				//
				txtInterfaceIPAddress.Enabled = false; 
				txtInterfaceDomain.Enabled = false;
				txtInterfacePort.Enabled = false;
			}
			else
			{
				rdoIpAddress.Enabled = true;
				rdoDomain.Enabled = true;
				//
				if (rdoIpAddress.Checked)
				{
					txtInterfaceIPAddress.Enabled = true;
				}
				else if (rdoDomain.Checked)
				{
					txtInterfaceDomain.Enabled = true;
				}
				//
				txtInterfacePort.Enabled = true;
			}

			if ( _currentConfig.Interface.InterfaceName == InterfaceName.BCE_COTS.ToString() ||
                _currentConfig.Interface.InterfaceName == InterfaceName.Auto_Instrument.ToString())
			{
				if (_currentConfig.Interface.InterfaceActiveIndicator == false)
				{
					cbInterfaceDisabled.Visible = true;
					cbInterfaceDisabled.Enabled = true;
					if (cbInterfaceDisabled.CheckState != CheckState.Checked)
					{
						_cbInterfaceDisabledCheckedByUser = false;
						cbInterfaceDisabled.CheckState = CheckState.Checked;
					}
				}
				else
				{
					cbInterfaceDisabled.Visible = true;
					cbInterfaceDisabled.Enabled = true;
					if (cbInterfaceDisabled.CheckState != CheckState.Unchecked)
					{
						_cbInterfaceDisabledCheckedByUser = false;
						cbInterfaceDisabled.CheckState = CheckState.Unchecked;
					}
				}
			}
			else
			{
				cbInterfaceDisabled.Visible = false;
			}
		}

		/// <summary>
		/// CR 2940
		/// </summary>
		private void btnSave_Click(object sender, EventArgs e)
		{
            // CR 3419
            if (!Utility.IsApplicationServerIP(txtVBECSIPAddress.Text))
            {
                if (GuiMessenger.ShowMessageBox(this, StrRes.ConfMsg.Common.Dynamic("The VBECS Application IP Address (" + txtVBECSIPAddress.Text + ") does not match the server's IP address (" + Utility.GetServerIPv4() + "). This is not a normal configuration. Do you still wish to save?")) != DialogResult.Yes)
                    return;
            }


            if (GuiMessenger.ShowMessageBox(this, StrRes.ConfMsg.Common.Dynamic("OK to save interface changes? If you have updated the service parameters, the service will also be restarted at this time.")) != DialogResult.Yes)
                return;

            _currentConfig.Save();
            // CR 3419: reset the flag
            _interfaceChanged = false;
            _upDownChanged = false;

            // CR 3419 start: if the services configuration changed, we must restart the services
            if (_serviceIpOrPortChanged)
            {
                bool success = false;
                if (VbecsConfig.Current.EnvironmentType.ToUpper() == "P")
                {
                    success = Utility.RestartService(true, ApplicationNameGlobalConfig.HL7Service);
                }
                else
                {
                    success = Utility.RestartService(false, ApplicationNameGlobalConfig.HL7Service);
                }

                if (!success) GuiMessenger.ShowMessageBox(this, StrRes.SysErrMsg.Common.Dynamic("The service failed to restart. Please contact VBECS support to have the service restarted."));
                _serviceIpOrPortChanged = false;
            }

            if (_interfaceChanged)
            {
                bool success = false;
                if (VbecsConfig.Current.EnvironmentType.ToUpper() == "P")
                {
                    success = Utility.RestartService(true, ApplicationNameGlobalConfig.HL7Client);
                }
                else
                {
                    success = Utility.RestartService(false, ApplicationNameGlobalConfig.HL7Client);
                }

                if (!success) GuiMessenger.ShowMessageBox(this, StrRes.SysErrMsg.Common.Dynamic("The service failed to restart. Please contact VBECS support to have the service restarted."));
                _serviceIpOrPortChanged = false;
            }

            // CR 3419 end

            if (_interfaceEnabled == true)
            {

                InterfaceControl.UpdateInterfaceStatus(_currentConfig.Interface.InterfaceControlId, true);
                _interfaceEnabled = false;
                if (_currentConfig.Interface.InterfaceName == InterfaceName.BCE_COTS.ToString())
                {
                    using (new WaitCursor())
                    {
                        VbecsBpsMessage.BceCotsInterfaceActivationUpdate(); // CR 3015
                    }
                }
            }
            UpdateButtons();
		}

		/// <summary>
		/// CR 2940
		/// </summary>
		private void DoInputDataTypeCheck()
		{
			//CR 2952, CR 2954 Added true to all calls to VerifyIPAddress since these values are required. This change will fix tooltips on Save button.
			//CR 2953 Changed the way _inputDataTypesAreTransferable is calculated
			if (_currentConfig.Interface.InterfaceName == InterfaceName.Patient_Merge.ToString() ||
				_currentConfig.Interface.InterfaceName == InterfaceName.Patient_Update.ToString() ||
                _currentConfig.Interface.InterfaceName == InterfaceName.Auto_Instrument.ToString())
			{
				_inputDataTypesAreTransferable = Utility.ForcedCheckLogicAnd(
 
					_validator.VerifyIPAddress( txtInterfaceIPAddress, false),
					_validator.VerifyPortNumber( txtInterfacePort, false ), 
					_validator.VerifyIPAddress(txtVBECSIPAddress, true),
					_validator.VerifyPortNumber(txtVBECSPort, true),
		
					_validator.VerifyInt32( updACKTimeout ),
					_validator.VerifyByte( updReTry ),

					_validator.VerifyByte( updCompletedMessages ),
					_validator.VerifyByte( updErrorMessages ) );
			}
			else if(_currentConfig.Interface.InterfaceName == InterfaceName.CPRS.ToString() ||
					_currentConfig.Interface.InterfaceName == InterfaceName.BCE_COTS.ToString())
			{
				_inputDataTypesAreTransferable = Utility.ForcedCheckLogicAnd(

					_validator.VerifyIPAddress( txtInterfaceIPAddress, rdoIpAddress.Checked == true),
					_validator.VerifyPortNumber( txtInterfacePort, true ), 
					_validator.VerifyIPAddress(txtVBECSIPAddress, true),
					_validator.VerifyPortNumber(txtVBECSPort, true),
		
					_validator.VerifyInt32( updACKTimeout ),
					_validator.VerifyByte( updReTry ),

					_validator.VerifyByte( updCompletedMessages ),
					_validator.VerifyByte( updErrorMessages ) );

				//CR 2949, CR 2952, CR 2954 Added check for Test Connection button
				_testConnectionCanBePerformed = Utility.ForcedCheckLogicAnd(
					_validator.VerifyIPAddress( txtInterfaceIPAddress, rdoIpAddress.Checked == true),
					_validator.VerifyPortNumber( txtInterfacePort, true ));

				//CR 2952, CR 2954 Added check for Domain tooltip
				if (rdoDomain.Checked == true)
				{
					if (txtInterfaceDomain.Text.Trim() == "")
					{
						_validator.SetValidationError(txtInterfaceDomain, StrRes.Tooltips.MUC04.InterfaceIPAddressMustBeSpecified());
						_inputDataTypesAreTransferable = false;
						//CR 2949
						_testConnectionCanBePerformed = false;
					}
					else
					{
						_validator.ClearValidationError(txtInterfaceDomain);
					}
				}
			}

			//CR 2952, CR 2954 Added check for Facility and Email tooltip
			if (txtInterfaceFacilityID.Text.Trim() == txtVBECSFacilityID.Text.Trim())
			{
				_validator.SetValidationError(txtInterfaceFacilityID, StrRes.Tooltips.MUC04.FacilitiesCannotMatch());
				_validator.SetValidationError(txtVBECSFacilityID, StrRes.Tooltips.MUC04.FacilitiesCannotMatch());
				_inputDataTypesAreTransferable = false;
			}
			else
			{
				_validator.ClearValidationError(txtInterfaceFacilityID);
				_validator.ClearValidationError(txtVBECSFacilityID);
			}

			if (RegularExpressions.EmailAddress().IsMatch(txtEmail.Text.Trim()) == false)
			{
				_validator.SetValidationError(txtEmail, StrRes.Tooltips.MUC04.EmailMustBeSpecified());
				_inputDataTypesAreTransferable = false;
			}
			else
			{
				_validator.ClearValidationError(txtEmail);
			}
		}

		/// <summary>
		/// 
		/// </summary>
		private void TransferInputToCurrentConfig()
		{	
			HL7Interface _interface = _currentConfig.Interface;
			
			try
			{
				_interface.InterfaceIPAddress = StUiNullConvert.ToIPAddress( txtInterfaceIPAddress.Text.Trim() );
			}
			catch (FormatException)
			{
				_interface.InterfaceIPAddress = null;
			}
			_interface.InterfaceDomainName = StUiNullConvert.ToString( txtInterfaceDomain.Text, true );
			_interface.InterfacePortNumber = StUiNullConvert.ToInt32( txtInterfacePort.Text );
			_interface.InterfaceFacilityId = StUiNullConvert.ToString( txtInterfaceFacilityID.Text, true );

			//CR 2936 - removing spaces from text
			_interface.VbecsIPAddress = StUiNullConvert.ToIPAddress( txtVBECSIPAddress.Text.Trim() );
			_interface.VbecsPortNumber = StUiNullConvert.ToInt32( txtVBECSPort.Text );
			_interface.VbecsFacilityId = StUiNullConvert.ToString( txtVBECSFacilityID.Text, true );

            try
            {   //CR3321 - updated to use VbecsNumericUpDown which throws a FormatException on .Get() if bad data
                _interface.AckTimeout = StUiNullConvert.DecimalToInt32(updACKTimeout.Value);
            }
            catch (FormatException)
            {
                _currentConfig.IsDirty = true;   //set this since they made an edit anyway, meh, but Stas's code sucks
            }

            try
            {   //CR3321 - updated to use VbecsNumericUpDown which throws a FormatException on .Get() if bad data
                _interface.ReTransmitAttempts = StUiNullConvert.DecimalToByte(updReTry.Value);
            }
            catch (FormatException)
            {
                _currentConfig.IsDirty = true;   //set this since they made an edit anyway, meh, but Stas's code sucks
            }

            try
            {   //CR3321 - updated to use VbecsNumericUpDown which throws a FormatException on .Get() if bad data
                _interface.PurgeSuccessfulMessage = StUiNullConvert.DecimalToByte(updCompletedMessages.Value);
            }
            catch (FormatException)
            {
                _currentConfig.IsDirty = true;   //set this since they made an edit anyway, meh, but Stas's code sucks
            }

            try
            {   //CR3321 - updated to use VbecsNumericUpDown which throws a FormatException on .Get() if bad data
                _interface.PurgeUnSuccessfulMessage = StUiNullConvert.DecimalToByte(updErrorMessages.Value);
            }
            catch (FormatException)
            {
                _currentConfig.IsDirty = true;   //set this since they made an edit anyway, meh, but Stas's code sucks
            }
			
			_interface.InterfaceAdministratorEmail = txtEmail.Text;

			_interface.LogEvents = cbLogEvents.Checked;
		}

		/// <summary>
		/// 
		/// </summary>
		private void NumericUpDown_KeyUp(object sender, KeyEventArgs e)
		{
			InputFieldChanged( sender, e );		
		}

		/// <summary>
		/// CR 2940
		/// </summary>
		private void InterfaceConnectivityInputFieldChanged( object sender, EventArgs e )
		{
			bool initialState = btnTestConnection.EnabledButton;
			//
			bool allowTestConnection = AllowConnectionTest();
			//
			btnTestConnection.EnabledButton = allowTestConnection;
			lblSuccessIndicator.Enabled = allowTestConnection;
			txtConnectionTestResults.Enabled = allowTestConnection;
			txtConnectionTestResults.Text = string.Empty;
			//
			btnSave.EnabledButton = false;
			//
			_connectionTestResult = InterfaceTestResult.NotTested;
			//
			lblSuccessIndicator.Image = imageListInterfaceTestResult.Images[0];
			//
            _interfaceChanged = true; // CR 3419
            //
			InputFieldChanged( sender, e );
		}


        /// <summary>
        /// CR 3419
        /// </summary>
        private void ServiceConnectivityInputFieldChanged(object sender, EventArgs e)
        {
            if (DataLoadMode.IsLocked) // To prevent processing of events when data is being transferred to widgets.
            {
                return;
            }
            //
            _serviceIpOrPortChanged = true;
            InputFieldChanged(sender, e);
        }


		/// <summary>
		/// 
		/// </summary>
		private void InputFieldChanged( object sender, EventArgs e )
		{
			if( DataLoadMode.IsLocked ) // To prevent processing of events when data is being transferred to widgets.
			{
				return;
			}

            // CR 3436
            if (sender is VbecsNumericUpDown && !_upDownChanged)
            {
                return;
            }

			//
			DoInputDataTypeCheck();			
			//
			if( _inputDataTypesAreTransferable )
			{
				TransferInputToCurrentConfig();
			
				_hl7GuiValidator.Validate( _currentConfig );				
			}
			//
			_currentConfig.IsDirty = true;
			UpdateButtons();
		}

        private void NumericUpDownValueChanged( object sender, EventArgs e )
        {
            _upDownChanged = true; // CR 3436
            InputFieldChanged(sender, e); 
        }







		private void UpdateSaveButtonState()
		{
			if ( _currentConfig.Interface.InterfaceName == InterfaceName.BCE_COTS.ToString() ||
				_currentConfig.Interface.InterfaceName == InterfaceName.CPRS.ToString())
			{
                // CR 3419 start
                if (_interfaceChanged)
                {
                    btnSave.EnabledButton = _inputDataTypesAreTransferable && _currentConfig.IsValid && _connectionTestResult == InterfaceTestResult.Success;
                }
                else
                {
                    btnSave.EnabledButton = _inputDataTypesAreTransferable && _currentConfig.IsValid && _currentConfig.IsDirty;
                }
            }
			else if ( _currentConfig.Interface.InterfaceName == InterfaceName.Patient_Merge.ToString() ||
					  _currentConfig.Interface.InterfaceName == InterfaceName.Patient_Update.ToString() ||
                      _currentConfig.Interface.InterfaceName == InterfaceName.Auto_Instrument.ToString() )
			{
				btnSave.EnabledButton = _inputDataTypesAreTransferable && _currentConfig.IsValid && _currentConfig.IsDirty;
			} // CR 3419 end
		}

		/// <summary>
		/// 
		/// </summary>
		private void UpdateSaveButtonToolTip()
		{
            string errorTT = "";

            //
            if (!_currentConfig.IsDirty)
            {
                SetToolTip(btnSave, StrRes.Tooltips.MUC04.NoChangesInConfig());
                return;
            }

            //CR 2952, CR 2954 
            if (!_currentConfig.IsValid || !_inputDataTypesAreTransferable)
            {
                string errorInterfaceIPAddress = _validator.GetValidationError(txtInterfaceIPAddress);
                if (errorInterfaceIPAddress != "")
                {
                    errorTT += errorInterfaceIPAddress;
                    errorTT += '\n';
                }

                string errorDomain = _validator.GetValidationError(txtInterfaceDomain);
                if (errorDomain != "")
                {
                    if (errorTT.IndexOf(errorDomain) == -1)
                    {
                        errorTT += errorDomain;
                        errorTT += '\n';
                    }
                }

                string errorVBECSIPAddress = _validator.GetValidationError(txtVBECSIPAddress);
                if (errorVBECSIPAddress != "")
                {
                    errorTT += errorVBECSIPAddress;
                    errorTT += '\n';
                }

                string errorInterfacePortNumber = _validator.GetValidationError(txtInterfacePort);
                if (errorInterfacePortNumber != "")
                {
                    errorTT += errorInterfacePortNumber;
                    errorTT += '\n';
                }

                string errorVBECSPortNumber = _validator.GetValidationError(txtVBECSPort);
                if (errorVBECSPortNumber != "")
                {
                    if (errorTT.IndexOf(errorVBECSPortNumber) == -1)
                    {
                        errorTT += errorVBECSPortNumber;
                        errorTT += '\n';
                    }
                }

                string errorFacility = _validator.GetValidationError(txtInterfaceFacilityID);
                if (errorFacility != "")
                {
                    errorTT += errorFacility;
                    errorTT += '\n';
                }

                string errorEmail = _validator.GetValidationError(txtEmail);
                if (errorEmail != "")
                {
                    errorTT += errorEmail;
                    errorTT += '\n';
                }

                SetToolTip(btnSave, errorTT);
                //return;
            }


            if (_currentConfig.Interface.InterfaceName == InterfaceName.BCE_COTS.ToString() ||
                _currentConfig.Interface.InterfaceName == InterfaceName.CPRS.ToString())
            {
                if (_interfaceChanged)
                {
                    if (_connectionTestResult == InterfaceTestResult.NotTested)
                    {
                        errorTT += _currentConfig.IsValid ? StrRes.Tooltips.MUC04.InterfaceConfigWasNotTestedAfterChange().ResString : _currentConfig.GetBrokenRulesMessage(BrRules.FormClassName);
                        btnSave.SetToolTip(errorTT);
                        SetToolTip(txtConnectionTestResults, StrRes.Tooltips.MUC04.InterfaceConfigWasNotTested());
                        SetToolTip(lblSuccessIndicator, StrRes.Tooltips.MUC04.InterfaceConfigWasNotTested());

                        return;
                    }
                    else if (_connectionTestResult == InterfaceTestResult.Testing)
                    {
                        errorTT += StrRes.Tooltips.MUC04.InterfaceConfigIsBeingTested().ResString;
                        return;
                    }
                    else if (_connectionTestResult == InterfaceTestResult.Success)
                    {
                        if (errorTT.Length == 0) // if there are already errors, we don't want to set the tooltip to nothing
                        {
                            errorTT += (_currentConfig.IsDirty ? StrRes.Tooltips.MUC04.ClickToSaveInterfaceConfig() : StrRes.Tooltips.MUC04.InterfaceConfigWasNotChanged()).ResString;
                            btnSave.SetToolTip(errorTT);
                        }
                        return;
                    }
                    else if (_connectionTestResult == InterfaceTestResult.Failure)
                    {
                        errorTT += StrRes.Tooltips.MUC04.InterfaceConnectionTestFailed().ResString;
                        btnSave.SetToolTip(errorTT);
                        return;
                    }
                }
            }
            //
            if (errorTT == string.Empty) SetToolTip(btnSave, StrRes.Tooltips.MUC04.ClickToSaveInterfaceConfig());
		}

		/// <summary>
		/// 
		/// </summary>
		private void btnClear_Click(object sender, EventArgs e)
		{
			_currentConfig.ReloadInterfaceFromDb();
			_currentConfig.IsDirty = false;
			//
			SwitchToInterfaceConfig( _currentConfig );
		}

		/// <summary>
		/// 
		/// </summary>
		private void rdo_CheckedChanged(object sender, EventArgs e)
		{
			if (rdoIpAddress.Checked)
			{
				txtInterfaceIPAddress.Enabled = true;
				txtInterfaceDomain.Enabled = false;
				txtInterfaceDomain.Text = string.Empty;

				_hl7GuiValidator.SetControlBindings( txtInterfaceIPAddress, BrRules.InterfaceIPAddressAndDomainNameNotSpecified );
				_hl7GuiValidator.SetControlBindings( txtInterfaceDomain, string.Empty );
				_validator.ClearValidationError(txtInterfaceDomain);
			}
			else
			{
				txtInterfaceDomain.Enabled = true;
				txtInterfaceIPAddress.Enabled = false;
				txtInterfaceIPAddress.Text = string.Empty;

				_hl7GuiValidator.SetControlBindings( txtInterfaceDomain, BrRules.InterfaceIPAddressAndDomainNameNotSpecified );
				_hl7GuiValidator.SetControlBindings( txtInterfaceIPAddress, string.Empty );
				_validator.ClearValidationError(txtInterfaceIPAddress);
			}
			InputFieldChanged(rdoIpAddress, new EventArgs());
		}

		/// <summary>
		/// CR 2940
		/// </summary>
		private void cbInterfaceDisabled_CheckStateChanged(object sender, EventArgs e)
		{
			if (_cbInterfaceDisabledCheckedByUser == false)
			{
				_cbInterfaceDisabledCheckedByUser = true;
				return;
			}

			if (cbInterfaceDisabled.CheckState == CheckState.Checked)
			{
                if (GuiMessenger.ShowMessageBox(this, StrRes.ConfMsg.MUC04.DisableInterface(_currentConfig.Interface.InterfaceName.Replace("_", " "))) == DialogResult.Yes)
				{
					InterfaceControl.UpdateInterfaceStatus(_currentConfig.Interface.InterfaceControlId, false);
					_currentConfig.Interface.InterfaceActiveIndicator = false;

					grpInterfacedApplication.Enabled = false;
					grpVBECSApplication.Enabled = false;
					grpMessageOptions.Enabled = false;
					grpPurgeCriteria.Enabled = false;
					grpRecipient.Enabled = false;
					grpEnableLogging.Enabled = false;
					btnClear.EnabledButton = false;
					btnSave.EnabledButton = false;
				}
				else
				{
					_cbInterfaceDisabledCheckedByUser = false;
					cbInterfaceDisabled.Checked = false;
				}
			}
			else
			{
                if (GuiMessenger.ShowMessageBox(this, StrRes.ConfMsg.MUC04.EnableInterface(_currentConfig.Interface.InterfaceName.Replace("_", " "))) == DialogResult.Yes)
				{
					_interfaceEnabled = true;
					_currentConfig.Interface.InterfaceActiveIndicator = true;

					grpInterfacedApplication.Enabled = true;
					grpVBECSApplication.Enabled = true;
					grpMessageOptions.Enabled = true;
					grpPurgeCriteria.Enabled = true;
					grpRecipient.Enabled = true;
					grpEnableLogging.Enabled = true;
					btnClear.EnabledButton = true;
					btnSave.EnabledButton = false;
					//CR 2950 set IsDirty flat so that the Save button enables when the interface is being enabled
					_currentConfig.IsDirty = true;
					DoInputDataTypeCheck();
					UpdateButtons();
				}
				else
				{
					_cbInterfaceDisabledCheckedByUser = false;
					cbInterfaceDisabled.Checked = true;
				}
			}
		}

		/// <summary>
		/// Set the tool tip text that appears when mouse hovers over the control
		/// </summary>
		private void txtInterfaceDomain_MouseHover(object sender, EventArgs e)
		{
			string finalcaption = txtInterfaceDomain.Text;	
			
			// Trying to avoid tooltip "flicker"
			if (finalcaption != toolTip1.GetToolTip(this))
			{
				toolTip1.SetToolTip(txtInterfaceDomain, finalcaption);
				toolTip1.SetToolTip(this, finalcaption);
			}
		}

		/// <summary>
		/// Set the tool tip text that appears when mouse hovers over the control
		/// </summary>
		private void txtVBECSFacilityID_MouseHover(object sender, EventArgs e)
		{
			string finalcaption = txtVBECSFacilityID.Text;	
			
			// Trying to avoid tooltip "flicker"
			if (finalcaption != toolTip1.GetToolTip(this))
			{
				toolTip1.SetToolTip(txtVBECSFacilityID, finalcaption);
				toolTip1.SetToolTip(this, finalcaption);
			}
		}

		/// <summary>
		/// Set the tool tip text that appears when mouse hovers over the control
		/// </summary>
		private void txtInterfaceFacilityID_MouseHover(object sender, EventArgs e)
		{
			string finalcaption = txtInterfaceFacilityID.Text;	
			
			// Trying to avoid tooltip "flicker"
			if (finalcaption != toolTip1.GetToolTip(this))
			{
				toolTip1.SetToolTip(txtInterfaceFacilityID, finalcaption);
				toolTip1.SetToolTip(this, finalcaption);
			}
		}
	}
}
